/*
	File:		UtilityWindow.cp

	Contains:	Utility window tester class.

	Version:	Appearance 1.0 SDK

	Copyright:	 1997 by Apple Computer, Inc., all rights reserved.

	File Ownership:

		DRI:				Edward Voas

		Other Contact:		7 of 9, Borg Collective

		Technology:			OS Technologies Group

	Writers:

		(edv)	Ed Voas

	Change History (most recent first):

		 <1>	 9/11/97	edv		First checked in.
*/

#include <MacWindows.h>
#include "UtilityWindow.h"
#include "Appearance.h"
#include "ColorPenState.h"
#include "Assertions.h"

#define width( r )		( (r).right - (r).left )
#define height( r )		( (r).bottom - (r).top )

UtilityWindow::UtilityWindow( SInt16 windID ) : BaseWindow( windID )
{				
	Rect		rect;
	
		// Note the use of the new defProc constants here.
		// This eliminates the need to go thru the mapping
		// CDEF for scroll bars, which would normally happen
		// after calling RegisterAppearanceClient.
		
	CalcVertScrollBarRect( rect );
	fVertScrollBar = NewControl( fWindow, &rect, "\p", true, 0, 0, 100, kControlScrollBarProc, 0 );
	
	CalcHorizScrollBarRect( rect );
	fHorizScrollBar = NewControl( fWindow, &rect, "\p", true, 0, 0, 100, kControlScrollBarProc, 0 );	
}

UtilityWindow::UtilityWindow() : BaseWindow( 131 )
{				
	Rect		rect;
	
		// Note the use of the new defProc constants here.
		// This eliminates the need to go thru the mapping
		// CDEF for scroll bars, which would normally happen
		// after calling RegisterAppearanceClient.
		
	CalcVertScrollBarRect( rect );
	fVertScrollBar = NewControl( fWindow, &rect, "\p", true, 0, 0, 100, kControlScrollBarProc, 0 );
	
	CalcHorizScrollBarRect( rect );
	fHorizScrollBar = NewControl( fWindow, &rect, "\p", true, 0, 0, 100, kControlScrollBarProc, 0 );	
}

UtilityWindow::~UtilityWindow()
{
	if ( fVertScrollBar )
		DisposeControl( fVertScrollBar );
	
	if ( fHorizScrollBar )
		DisposeControl( fHorizScrollBar );
}

//
//	 Activate
//
//	Activates the contents of the window.
//
void
UtilityWindow::Activate( EventRecord& )
{
		// Here we use the new ActivateControl call. We
		// could have still used HiliteControl, but this
		// is much more straightforward. It also works
		// right with and without embedding, so it's a
		// real good idea to start using it.
		
	ActivateControl( fHorizScrollBar );
	ActivateControl( fVertScrollBar );
}

//
//	 Deactivate
//
//	Deactivates the contents of the window.
//
void
UtilityWindow::Deactivate( EventRecord& )
{
		// Here we use the new DeactivateControl call. We
		// could have still used HiliteControl, but this
		// is much more straightforward. It also works
		// right with and without embedding, so it's a
		// real good idea to start using it.
		
	DeactivateControl( fHorizScrollBar );
	DeactivateControl( fVertScrollBar );
}

//
//	 Draw
//
//	Draws our window.
//
void
UtilityWindow::Draw()
{
	UpdateControls( fWindow, fWindow->visRgn );
}

//
//	 Resize
//
//	Resize our window to the appropriate size specified in width and height. Make sure
//	the scroll bars are repositioned properly.
//
void
UtilityWindow::Resize( SInt16 width, SInt16 height )
{
	Rect		horizRect, vertRect;
	
	InvalidateScrollBars();
	
	BaseWindow::Resize( width, height );
	
	InvalidateScrollBars();
	
	CalcHorizScrollBarRect( horizRect );
	CalcVertScrollBarRect( vertRect );
	
	MoveControl( fHorizScrollBar, horizRect.left, horizRect.top );
	MoveControl( fVertScrollBar, vertRect.left, vertRect.top );
	SizeControl( fHorizScrollBar, width( horizRect ), height( horizRect ) );
	SizeControl( fVertScrollBar, width( vertRect ), height( vertRect ) );
}

//
//	 HandleClick
//
//	Simple routine to handle scroll bar tracking, even though they don't do anything.
//
void
UtilityWindow::HandleClick( EventRecord& event )
{
	ControlHandle		control;
	SInt16				part;
	Point				localPt;
	ControlActionUPP	actionProc;
	
	SetPort( fWindow );
	localPt = event.where;
	GlobalToLocal( &localPt );
	
	part = FindControl( localPt, fWindow, &control );
	switch ( part )
	{
		case kControlUpButtonPart:
		case kControlDownButtonPart:
		case kControlPageUpPart:
		case kControlPageDownPart:
			actionProc = NewControlActionProc( ScrollBarAction );
			TrackControl( control, localPt, actionProc );
			DisposeRoutineDescriptor( actionProc );
			break;
		
		case kControlIndicatorPart:
			TrackControl( control, localPt, (ControlActionUPP)-1L );
			break;
	}
}

//
//	 InvalidateScrollBars
//
//	Invalidates the scroll bar areas.
//
void
UtilityWindow::InvalidateScrollBars()
{
	Rect		tempRect;
	
	SetPort( fWindow );
	CalcHorizScrollBarRect( tempRect );
	InvalRect( &tempRect );
	EraseRect( &tempRect );
	
	CalcVertScrollBarRect( tempRect );
	InvalRect( &tempRect );
	EraseRect( &tempRect );
}

//
//	 CalcHorizScrollBarRect
//
//	Calculates the position where the horizontal scroll bar should be placed.
//
void
UtilityWindow::CalcHorizScrollBarRect( Rect& rect )
{
	rect = fWindow->portRect;
	rect.bottom++;
	rect.left = -1;
	rect.top = rect.bottom - 11;
	rect.right -= 9;
}

//
//	 CalcVertScrollBarRect
//
//	Calculates the position where the vertical scroll bar should be placed.
//
void
UtilityWindow::CalcVertScrollBarRect( Rect& rect )
{
	rect = fWindow->portRect;
	rect.right++;
	rect.left = rect.right - 11;
	rect.bottom -= 9;
	rect.top = -1;
}

//
//	 ScrollBarAction
//
//	A simple callback to give some feedback when clicking the scroll arrows or page
//	up/down areas.
//
pascal void
UtilityWindow::ScrollBarAction( ControlHandle control, SInt16 part )
{
	switch ( part )
	{
		case kControlUpButtonPart:
			if ( GetControlValue( control) > GetControlMinimum( control ) )
				SetControlValue( control, GetControlValue( control ) - 1 );
			break;
		
		case kControlDownButtonPart:
			if ( GetControlValue( control) < GetControlMaximum( control ) )
				SetControlValue( control, GetControlValue( control ) + 1 );
			break;
			
		case kControlPageUpPart:
			if ( GetControlValue( control) > GetControlMinimum( control ) )
				SetControlValue( control, GetControlValue( control ) - 10 );
			break;
			
		case kControlPageDownPart:
			if ( GetControlValue( control) < GetControlMaximum( control ) )
				SetControlValue( control, GetControlValue( control ) + 10 );
			break;
	}			
}

